/* 
   indent indent -bli0 -bfda -kr -psl lf_proximity.c
*/

#include <stdio.h>
#include <stdlib.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_proximity.h"

typedef struct lf_proximity_xbar_t {
    lf_xbar_t *xp;
    int count;
    int index;
} lf_proximity_xbar_t;


static int
lf_proximty_cmp(const void *key,
		const void *datum)
{
    lf_proximity_xbar_t *a = (lf_proximity_xbar_t *) key;
    lf_proximity_xbar_t *b = (lf_proximity_xbar_t *) datum;

    return a->xp->clos_level - b->xp->clos_level;
}

/*
 * create an array of xbars from the fabric sorted by increasing clos level
*/
static lf_proximity_xbar_t *
lf_proximity_alloc_xbars(lf_fabric_t * fp)
{
    int i;
    lf_proximity_xbar_t *pd;

    if (!fp->num_xbars)
	return 0;

    /*get space for the xbar list */
    pd = malloc(sizeof(lf_proximity_xbar_t) * fp->num_xbars);

    if (!pd)
	LF_ERROR(("malloc failed in lf_proximity_alloc_xbars"));

    /*copy xbars into list */
    for (i = 0; i < fp->num_xbars; i++) {
	pd[i].xp = fp->xbars[i];
	pd[i].count = 0;
    }

    /*sort list by clos level */
    qsort(pd, fp->num_xbars, sizeof(lf_proximity_xbar_t), lf_proximty_cmp);

    /*now that we're sorted, make the indexes */
    for (i = 0; i < fp->num_xbars; i++)
	pd[i].index = i;

    /*return sorted list, for caller to free */
    return pd;
  except:return 0;
}


/*
 * make an array of all of a fabric's nics. caller frees. total nic count
 * is returned in the parameter num_nics.
*/
static lf_nic_t **
lf_proximity_alloc_nics(lf_fabric_t * fp,
			int *num_nics)
{
    int i, j, nic_count;
    lf_nic_t **nics;

    /*first count all the nics */
    nic_count = *num_nics = 0;
    for (i = 0; i < fp->num_hosts; i++)
	nic_count += fp->hosts[i]->num_nics;

    if (!nic_count)
	return 0;

    /*now alloc space */
    nics = (lf_nic_t **) malloc(sizeof(lf_nic_t *) * nic_count);
    if (!nics)
	LF_ERROR(("malloc failed in lf_proximity_alloc_nics"));

    /*go back and collect nics */
    nic_count = 0;
    for (i = 0; i < fp->num_hosts; i++) {
	for (j = 0; j < fp->hosts[i]->num_nics; j++)
	    nics[nic_count++] = fp->hosts[i]->nics[j];
    }

    *num_nics = nic_count;
    return nics;

  except:return 0;
}

/*
 * search down from xp and mark any nics found as belonging to root.
 * return number of matches.
*/

static int
lf_proximity_search_down(lf_proximity_xbar_t * root,
			 lf_xbar_t * xp)
{
    int i;
    int nics_found = 0;

    for (i = 0; i < xp->num_ports; i++) {
	lf_xbar_t *xp2 = LF_XBAR(xp->topo_ports[i]);
	if (xp2 && (xp->link_state [i] == LF_LINK_STATE_UP || xp->link_state [i] == LF_LINK_STATE_UNKNOWN)) {
	    if (xp2->ln_type == LF_NODE_XBAR
		&& xp2->clos_level < xp->clos_level)
		nics_found += lf_proximity_search_down(root, xp2);
	    else if (xp2->ln_type == LF_NODE_NIC
		     && ((lf_nic_t *) xp2)->proximity_xbar == -1) {
		((lf_nic_t *) xp2)->proximity_xbar = root->index;
		nics_found++;
	    }
	}
    }
    return nics_found;
}


/*
 * search down all xbars from 'first' up to 'last' and mark matching nics
 * return count of matched nics.
*/

static int
lf_proximity_partition_nics(lf_nic_t ** nics,
			    int num_nics,
			    lf_proximity_xbar_t * xbars,
			    int num_xbars,
			    int first,
			    int last)
{
    int i, nics_found;

    /*clear nic xbar matches */
    for (i = 0; i < num_nics; i++)
	nics[i]->proximity_xbar = -1;

    /*for each xbar at this clos, search down and make any nic xbar matches */

    nics_found = 0;
    for (i = first; i < last && nics_found < num_nics; i++) {
	xbars[i].count = lf_proximity_search_down(&xbars[i], xbars[i].xp);
	nics_found += xbars[i].count;
    }

    return nics_found;
}

void
lf_proximity_free(lf_proximity_t * p)
{
    int i;
    if (!p)
	return;

    if (p->nics)
	free(p->nics);
    if (p->groups) {
	for (i = 0; i < p->num_levels; i++)
	    if (p->groups[i])
		free(p->groups[i]);
	free(p->groups);
    }
    if (p->levels)
	free(p->levels);
    free(p);
}

/*
 * alloc, initialize, and return a complicated data structure containing all the proximity information
 * to be freed w/ lf_proximity_free.
 */

lf_proximity_t *
lf_proximity_create(lf_fabric_t * fp)
{
    lf_proximity_t *p = 0;
    lf_proximity_xbar_t *xbars = 0;
    int i, j, k, first, last, level, num_groups, nics_found;

    p = (lf_proximity_t *) calloc(sizeof(*p), 1);

    if (!p)
	LF_ERROR(("malloc failed in lf_proximity_create"));

    /*alloc and initialize some work arrays for the computation */

    if (!(xbars = lf_proximity_alloc_xbars(fp))) {
	free(p);
	return 0;
    }

    if (!(p->nics = lf_proximity_alloc_nics(fp, &p->num_nics))
	|| !p->num_nics) {
	lf_proximity_free(p);
	free(xbars);
	return 0;
    }

    /*count number of different clos levels */
    p->num_levels = 1;
    for (i = 0; i < fp->num_xbars; i++) {
	if (i && xbars[i].xp->clos_level != xbars[i - 1].xp->clos_level)
	    p->num_levels++;
    }

    /*allocate bits and pieces of stuff. this may be more space than we need. */

    if (!(p->levels = calloc(sizeof(int), p->num_levels)))
	LF_ERROR(("malloc failed in lf_proximity_create"));

    if (!(p->groups = calloc(sizeof(int *), p->num_levels)))
	LF_ERROR(("malloc failed in lf_proximity_create"));

    /*step along each set of equal clos_level xbars and partition nics among them. */
    level = 0;
    p->num_levels = 0;

    for (first = 0; first < fp->num_xbars; first = last) {
	/*find last xbar in range */
	for (last = first;
	     last < fp->num_xbars
	     && xbars[last].xp->clos_level == xbars[first].xp->clos_level;
	     last++);

	/*partition the nics among xbars at this clos level */
	nics_found =
	    lf_proximity_partition_nics(p->nics, p->num_nics, xbars,
					fp->num_xbars, first, last);

	/*count just the non empty levels. */
	if (!nics_found)
	    continue;

	/*record the level */
	p->levels[level] = xbars[first].xp->clos_level;
	p->num_levels++;

	/*alloc at least enough space for all of the counts and indices at this level */
	if (!
	    (p->groups[level] =
	     calloc(sizeof(int), (1 + nics_found + last - first))))
	    LF_ERROR(("malloc failed in lf_proximity_create"));

	/*for each xbar at this level, collect the nics. save [][0] for the num_group count */
	j = 1;
	num_groups = 0;

	for (i = first; i < last; i++) {
	    if (xbars[i].count) {
		num_groups++;
		p->groups[level][j++] = xbars[i].count;

		for (k = 0; k < p->num_nics; k++) {
		    if (p->nics[k]->proximity_xbar == i)
			p->groups[level][j++] = k;
		}
	    }
	}
	p->groups[level][0] = num_groups;
	level++;
    }
    free(xbars);
    return p;

  except:
    if (xbars)
	free(xbars);
    lf_proximity_free(p);
    return 0;
}
